package gov.va.genisis2.dao.impl;

import static org.springframework.ldap.query.LdapQueryBuilder.query;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attributes;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.ldap.core.AttributesMapper;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.stereotype.Repository;

import gov.va.genisis2.dao.IUserLdapDao;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.vo.LdapUser;
import gov.va.genisis2.vo.LdapUserGroup;

@Repository
public class UserLdapDao implements IUserLdapDao {

	private static final Logger LOGGER = LoggerFactory.getLogger(UserLdapDao.class);
	
	private static final String LDAP_GROUP_BASE = "ou=Gen2_Roles";
	private static final String LDAP_GROUP_FILTER = "objectClass";
	private static final String LDAP_GROUP_CLASS_TYPE = "groupOfNames";
	private static final String LDAP_USER_BASE = "ou=People";
	private static final String LDAP_USER_FILTER = "uid";
	private static final String LDAP_ATTR_CN = "cn";
	private static final String LDAP_ATTR_MEMBER = "member";
	private static final String LDAP_ATTR_GECOS = "gecos";
	private static final String LDAP_ATTR_SN = "sn";
	private static final String LDAP_ATTR_UID = "uid";
	private static final String LDAP_ATTR_MAIL = "mail";
	private static final String LDAP_USERNAME_DELIMITER1 = "cn=";
	private static final String LDAP_USERNAME_DELIMITER2 = ",";
	
	@Autowired
	private LdapTemplate ldapTemplate;
	
	@Override
	public List<LdapUserGroup> getAllUsersByGroup() throws GenisisDAOException {
		List<LdapUserGroup> userGroups;
		try {
			userGroups = ldapTemplate.search(query().base(LDAP_GROUP_BASE).where(LDAP_GROUP_FILTER).is(LDAP_GROUP_CLASS_TYPE), new GroupAttributesMapper());
		} catch(Exception ex) {
			LOGGER.error("Exception occurred on getAllUsers.", ex);
			throw new GenisisDAOException("Exception occurred on getAllUsers.", ex);
		}
		
		return userGroups;
	}
	
	private class GroupAttributesMapper implements AttributesMapper<LdapUserGroup> {
		
		@Override
		public LdapUserGroup mapFromAttributes(Attributes attrs) throws NamingException {
			List<LdapUser> ldapUsers = new ArrayList<>();
			LdapUserGroup ldapGroup = new LdapUserGroup();
			ldapGroup.setGroupName((String) attrs.get(LDAP_ATTR_CN).get());

			NamingEnumeration<?> memberEnumeration = attrs.get(LDAP_ATTR_MEMBER).getAll();
			while (memberEnumeration.hasMore()) {
				String uid = getUserName(memberEnumeration.next().toString());
				List<LdapUser> list = ldapTemplate.search(query().base(LDAP_USER_BASE).where(LDAP_USER_FILTER).is(uid), new UserAttributesMapper());
				if (list != null && !list.isEmpty()) {
					ldapUsers.add(list.get(0));
				}
			}
			ldapGroup.setUsers(ldapUsers);

			return ldapGroup;
		}
		
		private String getUserName(String member) {
			if (null == member || member.isEmpty() || member.trim().isEmpty()) {
				return null;
			}
			
			return member.substring(member.indexOf(LDAP_USERNAME_DELIMITER1) + 3, member.indexOf(LDAP_USERNAME_DELIMITER2));
		}
	}
	
	private class UserAttributesMapper implements AttributesMapper<LdapUser> {
		
		@Override
		public LdapUser mapFromAttributes(Attributes attrs) throws NamingException {
			LdapUser user = new LdapUser();
			user.setFirstName(getFirstName((String) attrs.get(LDAP_ATTR_GECOS).get()));
			user.setLastName((String) attrs.get(LDAP_ATTR_SN).get());
			String userName = (String) attrs.get(LDAP_ATTR_UID).get();
			user.setUsername((null == userName || userName.isEmpty()) ? StringUtils.EMPTY : userName.toUpperCase());
			user.setEmailId((String) attrs.get(LDAP_ATTR_MAIL).get());

			return user;
		}

		private String getFirstName(String fullName) {
			String firstName;
			
			if (null == fullName || fullName.isEmpty() || fullName.trim().isEmpty()) {
				return null;
			}
			
			firstName = new StringTokenizer(fullName).nextToken();
			
			return firstName;
		}
	}
}
